// Copyright (c) Caliper Corporation 2003.  All Rights Reserved
// Written by Peter H. Van Demark - Caliper Corporation - February 2002

    // The toolboxes are for working with colors, fill styles, and line styles:
    // The style editors are toolboxes for working with colors, fill styles, and line styles. 
    // They can be opened directly, or by using the Style Editors toolbox, 
    // which has a button to open each toolbox. Each toolbox returns a compound 
    // variable that is the result of using that editor: a color, a fill style or a line style, 
    // respectively. Each toolbox also displays the text for the call to the appropriate 
    // GISDK function that will create the color, fill style or line style. 
    // You can copy the text and paste it into a resource file. 
    // In the Fill Editor and Line Editor toolboxes you can also paste in the text 
    // and load it to see what that style looks like.
    
    // To run these toolboxes:
    // To use the style editors, you can use the GIS Developer's 
    // Kit add-in to compile Style Editors.rsc to a UI Database in the 
    // program folder called Style Editors.dbd, then set up a new Dialog Box add-in, 
    // with "Style Editors" as the Description, Name and UI Database. 
    // Starting this add-in displays the Style Editors toolbox.

// ---------------------------- Style Editors --------------------------------

dBox "Style Editors" Toolbox

    // Start Color Editor
    Button "Color Editor" 1, 1, 15  do
        color = RunDbox("Color Editor")
        EndItem

    // Start Fill Editor
    Button "Fill Editor" same, 2.75, 15  do
        fill_style = RunDbox("Fill Editor")
        EndItem

    // Start Line Editor
    Button "Line Editor" same, 4.5, 15  do
        line_style = RunDbox("Line Editor")
        EndItem

    // Close toolbox
    Close do
        Return()
        endItem

endDbox

// ---------------------------- Color Editor ---------------------------------

dBox "Color Editor" Toolbox

    init do
        colorsystem = 1
        limits = {65535, 240}
        RunMacro ("Clear Color")
        solidline = LineStyle({{{1, -1, 0}}})
        str1 = "XXXXXXXX"
        solidfill = FillStyle({str1, str1, str1, str1, str1, str1, str1, str1})
        prompts = {{"Red", "Green", "Blue"}, {"Hue", "Sat.", "Lum."}}
        endItem  

    // Clear color
    Macro "Clear Color" do
        bands = {{"0", "0", "0"}, {"0", "0", "0"}}
        vals = {{0, 0, 0}, {0, 0, 0}}
        dim lists[3]
        RunMacro("Set Color")
        endItem
    
    // Get color system
    Radio List "Color System" 0, 0, 27.5, 2 Prompt: "Color System"
        Variable: colorsystem
    Radio Button 5,1 Prompt: "RGB" do
        c = ColorHLS(vals[2][1], vals[2][3], vals[2][2])
        bands[1][1] = i2s(c.red)
        bands[1][2] = i2s(c.green)
        bands[1][3] = i2s(c.blue)
        RunMacro("Set Color")
        endItem
    Radio Button 15, same Prompt: "HSL" do
        vals[2] = GetColorHLS(ColorRGB(vals[1][1], vals[1][2], vals[1][3]))
        // Get HSL strings from HLS values 
        bands[2][1] = r2s(Round(vals[2][1] * 240/360, 0))
        bands[2][2] = r2s(Round(vals[2][3] * 240, 0))
        bands[2][3] = r2s(Round(vals[2][2] * 240, 0))
        RunMacro("Set Color")
        endItem

    // Get red band/hue
    Spinner 7, 3, 9 Prompt: prompts[colorsystem][1] List: lists[1]
        Variable: bands[colorsystem][1] do
        band = 1
        RunMacro("Get Band")
        RunMacro("Set Color")
        endItem

    // Get green band/saturation
    Spinner same, 4.5, 9 Prompt: prompts[colorsystem][2] List: lists[2]
        Variable: bands[colorsystem][2] do
        band = 2
        RunMacro("Get Band")
        RunMacro("Set Color")
        endItem
    
    // Get blue band/luminence
    Spinner same, 6, 9 Prompt: prompts[colorsystem][3] List: lists[3]
        Variable: bands[colorsystem][3] do
        band = 3
        RunMacro("Get Band")
        RunMacro("Set Color")
        endItem

    // Set color band, handling hex, Windows values, and bad values
    Macro "Get Band" do
    
        // Get band to work on
        bandtext = bands[colorsystem][band]
        
        // Handle RGB
        if colorsystem = 1 then do
            first = lower(Left(bandtext,1))
            second = lower(Substring(bandtext, 2, 1))
            // Handle hex values, starting with 0x
            if first = "0" and second = "x" then do
                rest = Substring(bandtext, 3, )
                if rest <> null then do
                    num = HexToInt(rest)
                    if num = null then num = 0
                    end
                else num = 0
                end
            // Handle hex values, starting with just x
            else if first = "x" then do
                rest = Substring(bandtext, 2, )
                if rest <> null then do
                    num = HexToInt(rest)
                    if num = null then num = 0
                    end
                else num = 0
                end
            // Handle Windows color range of 0 -> 255
            else if first = "w" then do
                num = r2i(max(min(s2r(Substring(bandtext, 2, )), 255), 0)* 257)
                end
            // Handle regular values
            else num = r2i(max(min(s2r(bandtext), 65535), 0))
            end
        
        // Handle Windows Hue, Saturation and Luminence
        else if colorsystem = 2 then do
            if band = 1 then maxval = 239
            else maxval = 240
            num = r2i(max(min(s2r(Substring(bandtext, 2, )), maxval), 0))
            end

        // Store result
        vals[colorsystem][band] = num
        endItem
    
    // Get resulting color, create sample, and create call to ColorRGB()
    Macro "Set Color" do
        
        // If Windows Hue, Saturation and Luminence (HLS) convert to RGB
        if colorsystem = 2 then do
            vals[2][1] = s2i(bands[2][1]) /240*360
            vals[2][2] = s2i(bands[2][2]) /240
            vals[2][3] = s2i(bands[2][3]) /240
            c = ColorHLS(vals[2][1], vals[2][3], vals[2][2])
            vals[1][1] = c.red
            vals[1][2] = c.green
            vals[1][3] = c.blue
            end
        
        // Create fill color, sample and text of call to ColorRGB()
        fillcolor = ColorRGB(vals[1][1], vals[1][2], vals[1][3])
        samplecolor = SampleArea(0, solidline, fillcolor, solidfill, fillcolor,)
        colortext = "ColorRGB(" + i2s(vals[1][1]) + "," + i2s(vals[1][2]) +
            "," + i2s(vals[1][3]) + ")"
        
        // Build spinner lists
        for band = 1 to 3 do
            num = s2r(bands[colorsystem][band])
            bandlist = {r2s(num)}
            if num - 1 ge 0 then bandlist = {r2s(num - 1)} + bandlist
            if num + 1 le limits[colorsystem] then
                bandlist = bandlist + {r2s(num + 1)}
            lists[band] = bandlist
            end
        endItem

    // Show sample
    Sample after, 3, 10, 4 Contents: samplecolor

    // Show call to ColorRGB()
    Edit Text 1, 7.5, 26 Variable: colortext Disabled
   
    // Clear color
    Button "Clear" 1, 9, 7.5  do
        RunMacro("Clear Color")
        EndItem
    
    // Force showing of results
    Button "Show" 10.3, same, 7.5  do
        RunMacro("Set Color")
        EndItem

    // Done
    Button "Close" 19.75, same, 7.5 cancel do
        Return(fillcolor)
        EndItem

    // Close toolbox
    Close do
        Return(fillcolor)
        endItem

endDbox

// ---------------------------- Fill Editor -----------------------------

dBox "Fill Editor" Toolbox

    init do
        Black = ColorRGB(0, 0, 0)
        White = ColorRGB(65535, 65535, 65535)
        colors = {White, Black}
        RunMacro ("Clear Fill")
       EndItem

    // Clear fill
    Macro "Clear Fill" do
        dim rowcol[8], samples[8]
        for i = 1 to 8 do
            rowcol[i] = {0, 0, 0, 0, 0, 0, 0, 0}
            for j = 1 to 8 do
                samples[i] =  samples[i] + {SampleArea( , , , solidfill, White,)}
                end
            end
        RunMacro("Set Fill")
        endItem

    // Use array of sample buttons to get fill style
    // Row #1
    Sample Button 1, 0, 1, .5 transparent contents: samples[1][1] do
        row = 1
        col = 1
        RunMacro("Set Cell")
        endItem
    Sample Button after, same, 1, .5 transparent contents: samples[1][2] do
        row = 1
        col = 2
        RunMacro("Set Cell")
        endItem
    Sample Button after, same, 1, .5 transparent contents: samples[1][3] do
        row = 1
        col = 3
        RunMacro("Set Cell")
       endItem
    Sample Button after, same, 1, .5 transparent contents: samples[1][4] do
        row = 1
        col = 4
        RunMacro("Set Cell")
        endItem
    Sample Button after, same, 1, .5 transparent contents: samples[1][5] do
        row = 1
        col = 5
        RunMacro("Set Cell")
       endItem
    Sample Button after, same, 1, .5 transparent contents: samples[1][6] do
        row = 1
        col = 6
        RunMacro("Set Cell")
        endItem
    Sample Button after, same, 1, .5 transparent contents: samples[1][7] do
        row = 1
        col = 7
        RunMacro("Set Cell")
        endItem
    Sample Button after, same, 1, .5 transparent contents: samples[1][8] do
        row = 1
        col = 8
        RunMacro("Set Cell")
        endItem

    // Row #2 
    Sample Button 1, after, 1, .5 transparent contents: samples[2][1] do
        row = 2
        col = 1
        RunMacro("Set Cell")
        endItem
    Sample Button after, same, 1, .5 transparent contents: samples[2][2] do
        row = 2
        col = 2
        RunMacro("Set Cell")
        endItem
    Sample Button after, same, 1, .5 transparent contents: samples[2][3] do
        row = 2
        col = 3
        RunMacro("Set Cell")
        endItem
    Sample Button after, same, 1, .5 transparent contents: samples[2][4] do
        row = 2
        col = 4
        RunMacro("Set Cell")
        endItem
    Sample Button after, same, 1, .5 transparent contents: samples[2][5] do
        row = 2
        col = 5
        RunMacro("Set Cell")
        endItem
    Sample Button after, same, 1, .5 transparent contents: samples[2][6] do
        row = 2
        col = 6
        RunMacro("Set Cell")
        endItem
    Sample Button after, same, 1, .5 transparent contents: samples[2][7] do
        row = 2
        col = 7
        RunMacro("Set Cell")
        endItem
    Sample Button after, same, 1, .5 transparent contents: samples[2][8] do
        row = 2
        col = 8
        RunMacro("Set Cell")
        endItem

    // Row #3 
    Sample Button 1, after, 1, .5 transparent contents: samples[3][1] do
        row = 3
        col = 1
        RunMacro("Set Cell")
        endItem
    Sample Button after, same, 1, .5 transparent contents: samples[3][2] do
        row = 3
        col = 2
        RunMacro("Set Cell")
        endItem
    Sample Button after, same, 1, .5 transparent contents: samples[3][3] do
        row = 3
        col = 3
        RunMacro("Set Cell")
        endItem
    Sample Button after, same, 1, .5 transparent contents: samples[3][4] do
        row = 3
        col = 4
        RunMacro("Set Cell")
        endItem
    Sample Button after, same, 1, .5 transparent contents: samples[3][5] do
        row = 3
        col = 5
        RunMacro("Set Cell")
        endItem
    Sample Button after, same, 1, .5 transparent contents: samples[3][6] do
        row = 3
        col = 6
        RunMacro("Set Cell")
        endItem
    Sample Button after, same, 1, .5 transparent contents: samples[3][7] do
        row = 3
        col = 7
        RunMacro("Set Cell")
        endItem
    Sample Button after, same, 1, .5 transparent contents: samples[3][8] do
        row = 3
        col = 8
        RunMacro("Set Cell")
        endItem

    // Row #4 
    Sample Button 1, after, 1, .5 transparent contents: samples[4][1] do
        row = 4
        col = 1
        RunMacro("Set Cell")
        endItem
    Sample Button after, same, 1, .5 transparent contents: samples[4][2] do
        row = 4
        col = 2
        RunMacro("Set Cell")
        endItem
    Sample Button after, same, 1, .5 transparent contents: samples[4][3] do
        row = 4
        col = 3
        RunMacro("Set Cell")
       endItem
    Sample Button after, same, 1, .5 transparent contents: samples[4][4] do
        row = 4
        col = 4
        RunMacro("Set Cell")
        endItem
    Sample Button after, same, 1, .5 transparent contents: samples[4][5] do
        row = 4
        col = 5
        RunMacro("Set Cell")
        endItem
    Sample Button after, same, 1, .5 transparent contents: samples[4][6] do
        row = 4
        col = 6
        RunMacro("Set Cell")
        endItem
    Sample Button after, same, 1, .5 transparent contents: samples[4][7] do
        row = 4
        col = 7
        RunMacro("Set Cell")
        endItem
    Sample Button after, same, 1, .5 transparent contents: samples[4][8] do
        row = 4
        col = 8
        RunMacro("Set Cell")
        endItem

    // Row #5 
    Sample Button 1, after, 1, .5 transparent contents: samples[5][1] do
        row = 5
        col = 1
        RunMacro("Set Cell")
        endItem
    Sample Button after, same, 1, .5 transparent contents: samples[5][2] do
        row = 5
        col = 2
        RunMacro("Set Cell")
       endItem
    Sample Button after, same, 1, .5 transparent contents: samples[5][3] do
        row = 5
        col = 3
        RunMacro("Set Cell")
        endItem
    Sample Button after, same, 1, .5 transparent contents: samples[5][4] do
        row = 5
        col = 4
        RunMacro("Set Cell")
        endItem
    Sample Button after, same, 1, .5 transparent contents: samples[5][5] do
        row = 5
        col = 5
        RunMacro("Set Cell")
        endItem
    Sample Button after, same, 1, .5 transparent contents: samples[5][6] do
        row = 5
        col = 6
        RunMacro("Set Cell")
        endItem
    Sample Button after, same, 1, .5 transparent contents: samples[5][7] do
        row = 5
        col = 7
        RunMacro("Set Cell")
        endItem
    Sample Button after, same, 1, .5 transparent contents: samples[5][8] do
        row = 5
        col = 8
        RunMacro("Set Cell")
        endItem

    // Row #6 
    Sample Button 1, after, 1, .5 transparent contents: samples[6][1] do
        row = 6
        col = 1
        RunMacro("Set Cell")
        endItem
    Sample Button after, same, 1, .5 transparent contents: samples[6][2] do
        row = 6
        col = 2
        RunMacro("Set Cell")
        endItem
    Sample Button after, same, 1, .5 transparent contents: samples[6][3] do
        row = 6
        col = 3
        RunMacro("Set Cell")
        endItem
    Sample Button after, same, 1, .5 transparent contents: samples[6][4] do
        row = 6
        col = 4
        RunMacro("Set Cell")
        endItem
    Sample Button after, same, 1, .5 transparent contents: samples[6][5] do
        row = 6
        col = 5
        RunMacro("Set Cell")
       endItem
    Sample Button after, same, 1, .5 transparent contents: samples[6][6] do
        row = 6
        col = 6
        RunMacro("Set Cell")
        endItem
    Sample Button after, same, 1, .5 transparent contents: samples[6][7] do
        row = 6
        col = 7
        RunMacro("Set Cell")
        endItem
    Sample Button after, same, 1, .5 transparent contents: samples[6][8] do
        row = 6
        col = 8
        RunMacro("Set Cell")
        endItem

    // Row #7 
    Sample Button 1, after, 1, .5 transparent contents: samples[7][1] do
        row = 7
        col = 1
        RunMacro("Set Cell")
        endItem
    Sample Button after, same, 1, .5 transparent contents: samples[7][2] do
        row = 7
        col = 2
        RunMacro("Set Cell")
        endItem
    Sample Button after, same, 1, .5 transparent contents: samples[7][3] do
        row = 7
        col = 3
        RunMacro("Set Cell")
        endItem
    Sample Button after, same, 1, .5 transparent contents: samples[7][4] do
        row = 7
        col = 4
        RunMacro("Set Cell")
        endItem
    Sample Button after, same, 1, .5 transparent contents: samples[7][5] do
        row = 7
        col = 5
        RunMacro("Set Cell")
        endItem
    Sample Button after, same, 1, .5 transparent contents: samples[7][6] do
        row = 7
        col = 6
        RunMacro("Set Cell")
        endItem
    Sample Button after, same, 1, .5 transparent contents: samples[7][7] do
        row = 7
        col = 7
        RunMacro("Set Cell")
        endItem
    Sample Button after, same, 1, .5 transparent contents: samples[7][8] do
        row = 7
        col = 8
        RunMacro("Set Cell")
        endItem

    // Row #8 
    Sample Button 1, after, 1, .5 transparent contents: samples[8][1] do
        row = 8
        col = 1
        RunMacro("Set Cell")
        endItem
    Sample Button after, same, 1, .5 transparent contents: samples[8][2] do
        row = 8
        col = 2
        RunMacro("Set Cell")
        endItem
    Sample Button after, same, 1, .5 transparent contents: samples[8][3] do
        row = 8
        col = 3
        RunMacro("Set Cell")
        endItem
    Sample Button after, same, 1, .5 transparent contents: samples[8][4] do
        row = 8
        col = 4
        RunMacro("Set Cell")
        endItem
    Sample Button after, same, 1, .5 transparent contents: samples[8][5] do
        row = 8
        col = 5
        RunMacro("Set Cell")
        endItem
    Sample Button after, same, 1, .5 transparent contents: samples[8][6] do
        row = 8
        col = 6
        RunMacro("Set Cell")
        endItem
    Sample Button after, same, 1, .5 transparent contents: samples[8][7] do
        row = 8
        col = 7
        RunMacro("Set Cell")
        endItem
    Sample Button after, same, 1, .5 transparent contents: samples[8][8] do
        row = 8
        col = 8
        RunMacro("Set Cell")
        endItem

    // Set cell
    Macro "Set Cell" do
        rowcol[row][col] = not rowcol[row][col]
        samples[row][col] = SampleArea( , , , solidfill,
            colors[rowcol[row][col] + 1],)
        RunMacro("Set Fill")        
        endItem
    
    // Get resulting fill, create sample, and create call to FillStyle()
    Macro "Set Fill" do
        dim rows[8]
        for i=1 to 8 do
            rows[i] = null
            for j = 1 to 8 do
                rows[i] = rows[i] + (if rowcol[i][j] then "x" else " ")
                end
            end
        samplefill = SampleArea ( , , , FillStyle(rows), Black, )
        filltext = "FillStyle({" 
        for i = 1 to 8 do
            filltext = filltext + '"' + rows[i] + '"'
            if i <> 8 then filltext = filltext + ","
            end
        filltext = filltext +  "})"
        oldfilltext = filltext
        endItem

    // Show sample
    Sample after, 0 , 16, 7  Contents: samplefill

    // Show call to FillStyle()
    Edit Text 1, 7.75, 33, 4 Variable: filltext
   
    // Load fill
    Button "Load" 3, after, 9  do
        pos1 = Position(filltext, "(")
        pos2 = Position(filltext, ")")
        rows = RunMacro("Text to Spec", Substring(filltext, pos1 + 1, pos2 - pos1 - 1))
        // Check for bad fill style
        on error do
            ShowMessage("Bad fill style specification")
            filltext = oldfilltext
            goto skip
            end
        temp = FillStyle(rows)     
        on error default
        // Reset cell flags and samples
        dim rowcol[8], samples[8]
        for i=1 to 8 do
            for j = 1 to 8 do
                rowcol[i] = rowcol[i] + {if Substring(rows[i], j, 1) = " "
                    then 0 else 1}
                samples[i] =  samples[i] + {SampleArea( , , , solidfill,
                    colors[rowcol[i][j] + 1],)}
                end
            end
        RunMacro("Set Fill")
skip:
        EndItem

    // Clear fill
    Button "Clear" 14, same, 9  do
        RunMacro("Clear Fill")
        EndItem

    // Done
    Button "Close" 25, same, 9 cancel do
        Return()
        EndItem

    // Close toolbox
    Close do
        Return(FillStyle(rows))
        endItem

endDbox

// Convert line style specification text to an array
Macro "Text to Spec" (atext)
    // Open a temporary RSC file and write a macro to get the array
    tmp = GetTempFileName("gisdk\\toolbox\\*.rsc")
    f = OpenFile(tmp, "w")
    WriteLine(f, 'Macro "Get Array"')
    WriteLine(f, "tarray = " + atext)
    WriteLine(f, "Return(tarray)")
    WriteLine(f, "endMacro")
    CloseFile(f)

    // Compile the temporary resource
    parts = SplitPath(tmp)
    cmd_line = "gisdk\\toolbox\\rsccw.exe -n -c " +
        "-u gisdk\\toolbox\\exec_ui gisdk\\toolbox\\" + parts[3] +  parts[4]
    res = RunProgram(cmd_line,)

    // Run the temporary resource
    SetAlternateInterface("gisdk\\toolbox\\exec_ui")
    tarray = RunMacro("Get Array")
    SetAlternateInterface()
    DeleteFile(tmp)
    Return(tarray)
endMacro

// ---------------------------- Line Editor ---------------------------------

dBox "Line Editor" Toolbox

    init do
        widthtext = "4"
        RunMacro("Set Line Width")
        componentcolorlist = {"Underlying","Black","White","Red","Green","Blue",
            "Light Blue","Yellow"}
        parastart = {1, -1, 0}
        perpstart = {0, -1, 12, 25, 0, 3.5, 0, -4}
        RunMacro ("Clear Line")
        
        // Create array of sample colors
        linecolors = RunMacro("G30 setup colors")
        RunMacro ("Clear Color")
        str1 = "XXXXXXXX"
        solidfill = FillStyle({str1, str1, str1, str1, str1, str1, str1, str1})
        dim colormenu[linecolors.length]
        for i = 1 to linecolors.length do
            colormenu[i] = SampleArea(0,,, solidfill, linecolors[i],)
            end
/*
        // Create array of sample line styles
        linestyles = RunMacro("G30 setup line styles")
        dim stylemenu[linestyles.length]
        stylemenu[1] = "None"
        for i = 2 to linestyles.length do
            if i < 38 then wid = 1 else wid = 4
            stylemenu[i] = SampleLine(wid, linestyles[i], linecolors[15], )
            end
*/        
        // Set components and line
        para_idx = null
        RunMacro("Set Parallel Components")
        perp_idx = null
        RunMacro("Set Perpendicular Components")
        RunMacro("Set Line")
        endItem  

    // Clear color
    Macro "Clear Color" do
        color_idx = 15
        line_color = linecolors[color_idx]
        vals = {line_color.red, line_color.green, line_color.blue}
        RunMacro("Set Color")
        endItem

    // Clear line
    Macro "Clear Line" do
        para_idx = null
        parallels = {CopyArray(parastart)}
        RunMacro("Set Parallel Components")
        perp_idx = null
        perpendiculars = {CopyArray(perpstart)}
        RunMacro("Set Perpendicular Components")
        endItem
    
    // Get resulting line, create sample, and create text from arrays
    Macro "Set Line" do
        on error do
            seterror = True
            goto skip
            end
        line_style = LineStyle({parallels, perpendiculars})
        on error default

        sampleline = SampleLine(line_width, line_style, line_color, )
        // Create text arrays for scroll lists and text for call to LineStyle()
        linetext = "LineStyle({" 
        parallellist = null
        if parallels <> null then do
            {atext, parallellist} = RunMacro("Arrays to Text", parallels) 
            linetext = linetext + atext
            end
        perpendicularlist = null
        if perpendiculars <> null then do
            linetext = linetext + ", "
            {atext, perpendicularlist} =
                RunMacro("Arrays to Text", perpendiculars)
            linetext = linetext + atext
            end 
        linetext = linetext +"})"
skip:
        endItem

    // Show sample
    Sample 1, 16.5, 40, 3 Contents: sampleline

    // Show call to LineStyle()
    Edit Text 1, after, 40, 4 Variable: linetext 
   
    // Apply the line style, width and color to the current layer, if a line layer
    Button "Apply" 1, after, 6.5 do
        layer = GetLayer()
        if GetLayerType(layer) = "Line" then do
            layerset = layer + "|"
            SetLineStyle(layerset, line_style)
            SetLineColor(layerset, line_color)
            SetLineWidth(layerset, line_width)
            RedrawMap()
            end
        else if GetLayerType(layer) = "Area" then do
            layerset = layer + "|"
            SetBorderStyle(layerset, line_style)
            SetBorderColor(layerset, line_color)
            SetBorderWidth(layerset, line_width)
            RedrawMap()
            end
        else ShowMessage("Current layer not a line or area layer")
        endItem

    // Load settings
    Button "Load" 9.33, same, 6.5 do
        pos1 = Position(linetext, "(")
        pos2 = Position(linetext, ")")
        oldspec = {parallels, perpendiculars}
        {parallels, perpendiculars} = RunMacro("Text to Spec",
            Substring(linetext, pos1 + 1, pos2 - pos1 - 1))
        seterror = False
        RunMacro("Set Line")
        if seterror then do
            ShowMessage("Bad line style specification")
            parallels = oldspec[1]
            perpendiculars = oldspec[2]
            RunMacro("Set Line")
            end
        para_idx = null
        RunMacro("Set Parallel Components")
        perp_idx = null
        RunMacro("Set Perpendicular Components")
        EndItem

    // Clear settings
    Button "Clear" 17.66, same, 6.5 do
        RunMacro("Clear Line")
        RunMacro("Clear Color")
        RunMacro("Set Line")
        EndItem
    
    // Force showing of results
    Button "Show" 26, same, 6.5  do
        RunMacro("Set Line")
        EndItem

    // Done
    Button "Close" 34.33, same, 6.5 cancel do
        Return(line_style)
        EndItem

    // Close toolbox
    Close do
        Return(line_style)
        endItem

    // Tab list for sets of controls
    Tab List 1, 0 ,40 , 16 Variable: tab_idx
    
    // Controls for line settings
    Tab prompt: "Line"

	// Get line width
    Spinner 8, 1, 9 Prompt: "Width" List: widthlist
        Variable: widthtext do
        // Reset width list, limiting to 24 points
        RunMacro("Set Line Width")
        RunMacro("Set Line")
        endItem

    Macro "Set Line Width" do
        line_width = s2i(widthtext)
        widthlist = RunMacro("Build Spinner List",
            line_width, 1, 0, 24)
        endItem
	
/*    // Get line style
	Popdown Menu after, same, 9, 10 Prompt: "Style" List: stylemenu
        Variable: style_idx do
        line_style = linestyles[style_idx]
        {parallels, perpendiculars} = GetLineStyleSpec(line_style)
        // Reset line width appropriate for line style
        if style_idx < 38 then line_width = 1 else line_width = 4
        widthtext = i2s(line_width)
        widthlist = RunMacro("Build Spinner List",
            line_width, 1, 0, 24)
        RunMacro("Set Line")
        endItem
*/
	// Get line color
	Popdown Menu 8, after, 9, 10 Prompt: "Color" List: colormenu
        Variable: color_idx do
        line_color = linecolors[color_idx]
        vals = {line_color.red, line_color.green, line_color.blue}
        RunMacro("Set Color")
        RunMacro("Set Line")
        endItem

    // Frame for color editor
    Frame .5, after, 38, 8 Prompt: "Color Editor" 

    // Get red band
    Spinner 8, 6, 9 Prompt: "Red" List: lists[1]
        Variable: bands[1] do
        band = 1
        RunMacro("Get Band")
        RunMacro("Set Color")
        RunMacro("Set Line")
        lists[band] = RunMacro("Build Spinner List", vals[band], 1, 0, 65535)
        endItem

    // Get green band/saturation
    Spinner same, after, 9 Prompt: "Green" List: lists[2]
        Variable: bands[2] do
        band = 2
        RunMacro("Get Band")
        RunMacro("Set Color")
        RunMacro("Set Line")
        lists[band] = RunMacro("Build Spinner List", vals[band], 1, 0, 65535)
        endItem
    
    // Get blue band/luminence
    Spinner same, after, 9 Prompt: "Blue" List: lists[3]
        Variable: bands[3] do
        band = 3
        RunMacro("Get Band")
        RunMacro("Set Color")
        RunMacro("Set Line")
        lists[band] = RunMacro("Build Spinner List", vals[band], 1, 0, 65535)
        endItem

    // Set color band, handling hex, Windows values, and bad values
    Macro "Get Band" do
    
        // Get band to work on
        bandtext = bands[band]
        first = lower(Left(bandtext,1))
        second = lower(Substring(bandtext, 2, 1))
        // Handle hex values, starting with 0x
        if first = "0" and second = "x" then do
            rest = Substring(bandtext, 3, )
            if rest <> null then do
                num = HexToInt(rest)
                if num = null then num = 0
                end
            else num = 0
            end
        // Handle hex values, starting with  just x
        else if first = "x" then do
            rest = Substring(bandtext, 2, )
            if rest <> null then do
                num = HexToInt(rest)
                if num = null then num = 0
                end
            else num = 0
            end
        // Handle Windows color range of 0 -> 255
        else if first = "w" then do
            num = r2i(max(min(s2r(Substring(bandtext, 2, )), 255), 0)* 257)
            end
        // Handle regular values
        else num = r2i(max(min(s2r(bandtext), 65535), 0))
        vals[band] = num
        endItem
    
    // Get resulting color, create sample, call to ColorRGB(), and band lists
    Macro "Set Color" do
        line_color = ColorRGB(vals[1], vals[2], vals[3])
        samplecolor = SampleArea(0, solidline, fillcolor, solidfill, line_color,)
        colortext = "ColorRGB(" + i2s(vals[1]) + "," + i2s(vals[2]) +
            "," + i2s(vals[3]) + ")"
        dim bands[3], lists[3]
        for i = 1 to 3 do
            bands[i] = i2s(vals[i])
            lists[i] = RunMacro("Build Spinner List", vals[i], 1, 0, 65535)
             end
        endItem

    // Show sample
    Sample after, 6, 10, 4 Contents: samplecolor

    // Clear color
    Button "Clear" after, 6.5, 7.5  do
        RunMacro("Clear Color")
        RunMacro("Set Line")
        EndItem
    
    // Force showing of color
    Button "Show" same, 8.25, 7.5  do
        RunMacro("Set Color")
        RunMacro("Set Line")
        EndItem
	 
    // Show call to ColorRGB()
    Edit Text 2, 11, 26 Variable: colortext Disabled
   
    // Parallel controls
    Tab prompt: "Parallel"
    
    // List of parallel components
    Scroll List 1, .5, 36, 5  List: parallellist
    Variable: para_idx do
        RunMacro("Set Parallel Components")
        endItem

    Macro "Set Parallel Components" do
        dim paravars[4], paraspinlists[4]
        if para_idx <> null then do
            paravars[1] = r2s(parallels[para_idx][1])
            paraspinlists[1] = RunMacro("Build Spinner List",
                parallels[para_idx][1], 1, 0, 24)
            paravars[2] = r2i(parallels[para_idx][2]) + 2
            paravars[3] = r2s(parallels[para_idx][3])
            paraspinlists[3] = RunMacro("Build Spinner List",
                parallels[para_idx][3], 1, -12, 12)
            if parallels[para_idx].length > 3 then do
                paravars[4] = RunMacro("Array to List",
                    SubArray(parallels[para_idx], 4, ))
                end
            EnableItem("Parallel Copy")
            EnableItem("Parallel Delete")
            if para_idx = 1 then DisableItem("Parallel Move Up")
            else EnableItem("Parallel Move Up")
            if para_idx = parallels.length then DisableItem("Parallel Move Down")
            else EnableItem("Parallel Move Down")
            end
        else do
            DisableItem("Parallel Copy")
            DisableItem("Parallel Delete")
            DisableItem("Parallel Move Up")
            DisableItem("Parallel Move Down")
            end
        endItem

	// Get parallel component width
    Spinner 9, 6.25, 12 Prompt: "Width" List: paraspinlists[1]
        Variable: paravars[1] do
        if para_idx <> null then do
            parallels[para_idx][1] = s2r(paravars[1])
            paraspinlists[1] = RunMacro("Build Spinner List",
                parallels[para_idx][1], 1, 0, 24)
            RunMacro("Set Line")
            end
        endItem

	// Get parallel component color
	Popdown Menu same, 8.5, 12, 10 Prompt: "Color" List: componentcolorlist
        Variable: paravars[2] do
        if para_idx <> null then do
            parallels[para_idx][2] = paravars[2] - 2
            RunMacro("Set Line")
            end
        endItem

	// Get parallel component offset
    Spinner same, 10.75, 12 Prompt: "Offset" List: paraspinlists[3]
        Variable: paravars[3] do
        if para_idx <> null then do
            parallels[para_idx][3] = s2r(paravars[3])
            paraspinlists[3] = RunMacro("Build Spinner List",
                parallels[para_idx][3], 1, -12, 12)
            RunMacro("Set Line")
            end
        endItem

    // Get parallel component dashes
    Edit Text same, 13, 28  Prompt: "Dashes" Variable: paravars[4] do
        dashes = RunMacro("Text to Array",paravars[4])
        // Make sure dashes and spaces are in pairs
        if mod(dashes.length, 2) = 0 then do
            parallels[para_idx] = Subarray(parallels[para_idx], 1, 3) + dashes
            RunMacro("Set Line")
            end
        else ShowMessage("Dashes and spaces must be in pairs")    
        EndItem

    // Add a parallel component
    Button "Add" 26.5, 5.75, 10  do
        parallels = parallels + {CopyArray(parastart)}
        para_idx = parallels.length
        RunMacro("Set Parallel Components")
        RunMacro("Set Line")
        EndItem

    // Copy a parallel component
    Button "Parallel Copy" same, after, 10 Prompt: "Copy" do
        parallels = parallels + {CopyArray(parallels[para_idx])}
        para_idx = parallels.length
        RunMacro("Set Parallel Components")
        RunMacro("Set Line")
        EndItem

    // Delete a parallel component
    Button "Parallel Delete" same, after, 10 Prompt: "Delete" do
        parallels = ExcludeArrayElements(parallels, para_idx, 1)
        if parallels = null then para_idx = null
        else para_idx = r2i(max(para_idx - 1, 1))
        RunMacro("Set Parallel Components")
        RunMacro("Set Line")
        EndItem

    // Move a parallel component up
    Button "Parallel Move Up" same, after, 10 Prompt: "Move Up" do
        temp = null
        if para_idx > 2 then temp = Subarray(parallels, 1, para_idx - 2)
        temp = temp + {CopyArray(parallels[para_idx])} +
            {CopyArray(parallels[para_idx - 1])}
        if para_idx < parallels.length then temp = temp +
            Subarray(parallels, para_idx + 1, )
        parallels = temp
        para_idx = para_idx - 1
        RunMacro("Set Parallel Components")
        RunMacro("Set Line")
        EndItem

    // Move a parallel component down
    Button "Parallel Move Down" same, after, 10 Prompt: "Move Down" do
        temp = null
        if para_idx > 1 then temp = Subarray(parallels, 1, para_idx - 1)
        temp = temp + {CopyArray(parallels[para_idx + 1])} +
            {CopyArray(parallels[para_idx])}
        if para_idx + 1 < parallels.length then temp = temp +
            Subarray(parallels, para_idx + 2, )
        parallels = temp
        para_idx = para_idx + 1
        RunMacro("Set Parallel Components")
        RunMacro("Set Line")
        EndItem

    // Perpendicular controls
    Tab prompt: "Perpendicular"

    // List of perpendicular components
    Scroll List 1, .5, 36, 5  List: perpendicularlist
    Variable: perp_idx do
        RunMacro("Set Perpendicular Components")
        endItem

    Macro "Set Perpendicular Components" do
        dim perpvars[5], perpspinlists[5]
        if perp_idx <> null then do
            perpvars[1] = r2s(perpendiculars[perp_idx][1])
            perpspinlists[1] = RunMacro("Build Spinner List",
                perpendiculars[perp_idx][1], 1, 0, 24)
            perpvars[2] = r2i(perpendiculars[perp_idx][2]) + 2
            perpvars[3] = r2s(perpendiculars[perp_idx][3])
            perpspinlists[3] = RunMacro("Build Spinner List",
                perpendiculars[perp_idx][3], 1, 0, 72)
            perpvars[4] = r2s(perpendiculars[perp_idx][4])
            perpspinlists[4] = RunMacro("Build Spinner List",
                perpendiculars[perp_idx][4], 1, 0, 72)
            perpvars[5] = RunMacro("Array to List",
                SubArray(perpendiculars[perp_idx], 5, ))
            EnableItem("Perpendicular Copy")
            EnableItem("Perpendicular Delete")
            if perp_idx = 1 then DisableItem("Perpendicular Move Up")
            else EnableItem("Perpendicular Move Up")
            if perp_idx = perpendiculars.length then
                DisableItem("Perpendicular Move Down")
            else EnableItem("Perpendicular Move Down")
            end
        else do
            DisableItem("Perpendicular Copy")
            DisableItem("Perpendicular Delete")
            DisableItem("Perpendicular Move Up")
            DisableItem("Perpendicular Move Down")
            end
        endItem

	// Get perpendicular component width
    Spinner 10.5, after, 12 Prompt: "Width" List: perpspinlists[1]
        Variable: perpvars[1] do
        if perp_idx <> null then do
            perpendiculars[perp_idx][1] = s2r(perpvars[1])
            perpspinlists[1] = RunMacro("Build Spinner List",
                perpendiculars[perp_idx][1], 1, 0, 24)
            RunMacro("Set Line")
            end
        endItem

	// Get perpendicular component color
	Popdown Menu same, after, 12, 10 Prompt: "Color" List: componentcolorlist
        Variable: perpvars[2] do
        if perpvars[2] <> null then do
            perpendiculars[perp_idx][2] = perpvars[2] - 2
            RunMacro("Set Line")
            end
        endItem

	// Get perpendicular component distance from start
    Spinner same, after, 12 Prompt: "Distance" List: perpspinlists[3]
        Variable: perpvars[3] do
        if perp_idx <> null then do
            perpendiculars[perp_idx][3] = s2r(perpvars[3])
            perpspinlists[3] = RunMacro("Build Spinner List",
                perpendiculars[perp_idx][3], 1, 0, 72)
            RunMacro("Set Line")
            end
        endItem

	// Get perpendicular component frequency
    Spinner same, after, 12 Prompt: "Frequency" List: perpspinlists[4]
        Variable: perpvars[4] do
        if perp_idx <> null then do
            perpendiculars[perp_idx][4] = s2r(perpvars[4])
            perpspinlists[4] = RunMacro("Build Spinner List",
                perpendiculars[perp_idx][4], 1, 0, 72)
            RunMacro("Set Line")
            end
        endItem

    // Get perpendicular component XYs
    Edit Text same, after, 28  Prompt: "XYs" Variable: perpvars[5] do
        xys = RunMacro("Text to Array",perpvars[5])
        // Make sure there are 2 XY pairs
        if xys.length = 4 then do
            perpendiculars[perp_idx] = Subarray(perpendiculars[perp_idx], 1, 4) + xys
            RunMacro("Set Line")
            end
        else ShowMessage("There must be 2 XY pairs")    
        EndItem

    // Add a perpendicular component
    Button "Add" 26.5, 5.75, 10  do
        perpendiculars = perpendiculars + {CopyArray(perpstart)}
        perp_idx = perpendiculars.length
        RunMacro("Set Perpendicular Components")
        RunMacro("Set Line")
        EndItem

    // Copy a perpendicular component
    Button "Perpendicular Copy" same, after, 10 Prompt: "Copy" do
        if perp_idx <> null then do
            perpendiculars = perpendiculars + {CopyArray(perpendiculars[perp_idx])}
            perp_idx = perpendiculars.length
            RunMacro("Set Perpendicular Components")
            RunMacro("Set Line")
            end
        EndItem

    // Delete a perpendicular component
    Button "Perpendicular Delete" same, after, 10 Prompt: "Delete" do
        if perp_idx <> null then do
            perpendiculars = ExcludeArrayElements(perpendiculars, perp_idx, 1)
            if perpendiculars = null then perp_idx = null
            else perp_idx = r2i(max(perp_idx - 1, 1))
            RunMacro("Set Perpendicular Components")
            RunMacro("Set Line")
            end
        EndItem

    // Move a perpendicular component up
    Button "Perpendicular Move Up" same, after, 10 Prompt: "Move Up" do
        temp = null
        if perp_idx > 2 then temp = Subarray(perpendiculars, 1, perp_idx - 2)
        temp = temp + {CopyArray(perpendiculars[perp_idx])} +
            {CopyArray(perpendiculars[perp_idx - 1])}
        if perp_idx < perpendiculars.length then temp = temp +
            Subarray(perpendiculars, perp_idx + 1, )
        perpendiculars = temp
        perp_idx = perp_idx - 1
        RunMacro("Set Perpendicular Components")
        RunMacro("Set Line")
        EndItem

    // Move a perpendicular component down
    Button "Perpendicular Move Down" same, after, 10 Prompt: "Move Down" do
        temp = null
        if perp_idx > 1 then temp = Subarray(perpendiculars, 1, perp_idx - 1)
        temp = temp + {CopyArray(perpendiculars[perp_idx + 1])} +
            {CopyArray(perpendiculars[perp_idx])}
        if perp_idx + 1 < perpendiculars.length then temp = temp +
            Subarray(perpendiculars, perp_idx + 2, )
        perpendiculars = temp
        perp_idx = perp_idx + 1
        RunMacro("Set Perpendicular Components")
        RunMacro("Set Line")
        EndItem

endDbox

Macro "Build Spinner List" (val, step, low, high)
    spinnerlist = {r2s(val)}
    if val - step ge low then
        spinnerlist = {r2s(val - step)} + spinnerlist
    if val + step le high then
        spinnerlist = spinnerlist + {r2s(val + step)}
    Return(spinnerlist)
endMacro

// Change an array of arrays to text
Macro "Arrays to Text"(tarrays)
    atexts = "{"
    tlist = null
    if tarrays.length <> null then for i = 1 to tarrays.length do
        atext = RunMacro ("Array to List", tarrays[i])
        tlist = tlist + {atext} 
        atexts = atexts + "{" + atext + "}"
        if i <> tarrays.length then atexts = atexts + ", "
        end
    atexts = atexts + "}"
    Return({atexts, tlist})
endMacro

// Change an array to a list
Macro "Array to List"(tarray)
    atext = null
    if tarray.length <> null then for i = 1 to tarray.length do
        if TypeOf(tarray[i])then atext = atext + i2s(tarray[i])
        else atext = atext + r2s(tarray[i])
        if i <> tarray.length then atext = atext + ", "
        end
    Return(atext)
endMacro

// Change a text list of values to an array
Macro "Text to Array" (atext)
    tarray = null
    if atext <> null then do
        pieces = ParseString(atext, ", ")
        for i = 1 to pieces.length do
            tarray = tarray + {s2r(pieces[i])}
            end
        end
    Return(tarray)
endMacro

// Convert line style specification text to an array
Macro "Text to Spec" (atext)
    // Open a temporary RSC file and write a macro to get the array
    tmp = GetTempFileName("gisdk\\toolbox\\*.rsc")
    f = OpenFile(tmp, "w")
    WriteLine(f, 'Macro "Get Array"')
    WriteLine(f, "tarray = " + atext)
    WriteLine(f, "Return(tarray)")
    WriteLine(f, "endMacro")
    CloseFile(f)

    // Compile the temporary resource
    parts = SplitPath(tmp)
    cmd_line = "gisdk\\toolbox\\rsccw.exe -n -c " +
        "-u gisdk\\toolbox\\exec_ui gisdk\\toolbox\\" + parts[3] +  parts[4]
    res = RunProgram(cmd_line,)

    // Run the temporary resource
    SetAlternateInterface("gisdk\\toolbox\\exec_ui")
    tarray = RunMacro("Get Array")
    SetAlternateInterface()
    DeleteFile(tmp)
    Return(tarray)
endMacro
